
/* Copyright (C) 2001-2008 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_cmap.c */

#include "fs_itype.h"
#ifdef FS_EDGE_RENDER
#include "adf.h"
#endif

/* local prototypes */
static FS_USHORT map_char_0(FS_USHORT *cmap, FS_ULONG id);
static FS_USHORT map_char_2(FS_USHORT *cmap, FS_ULONG id);
static FS_USHORT map_char_4(FS_USHORT *cmap, FS_ULONG id);
static FS_USHORT map_char_5(FS_USHORT *cmap, FS_ULONG id);
static FS_USHORT map_char_6(FS_USHORT *cmap, FS_ULONG id);
static FS_USHORT map_char_8(FS_USHORT *cmap, FS_ULONG id);
static FS_USHORT map_char_12(FS_USHORT *cmap, FS_ULONG id);
static FS_USHORT map_char_14(FS_USHORT *cmap, FS_ULONG id, FS_ULONG varSelector);
static FS_ULONG inverse_map_char_4(FS_USHORT *cmap, FS_USHORT gi);
static FS_ULONG inverse_map_char_12(FS_USHORT *cmap, FS_ULONG gi);
#ifdef FS_ICONS
static FS_USHORT map_char_icon(_DS_ TTF *ttf, FS_ULONG id);
#endif /* FS_ICONS */


/* 16807 is a popular multiplier in linear congurential
 * random number generators. (eg: Parker-Miller MINSTD).
 *
 * 16807 is the smallest number such that
 * power(16807,N) fully covers the range 0..2^31 - 1
 *
 * in this application it's also a 16*16=>32 multiply
 * which almost all modern chips can do quite quickly
 */
#define CMAP_CACHE_SLOT(lfnt,id) (FS_USHORT)((16807 * (id)) & (lfnt)->cmap_cache_mod)


static FS_ULONG
LFNT_get_cmap_offset(_DS_ LFNT *lfnt, FS_USHORT platform, FS_USHORT encoding)
{
    TTF *ttf;
    TTF_CMAP *cmap;
    FS_LONG i, n_tabs;
    CMAP_TAB *tabs;
    FS_ULONG offset;

#ifdef FS_PFRR
    if (lfnt->fnt_type == PFR_TYPE)
    {
        STATE.error = SUCCESS;
        return 0;
    }
#endif /* FS_PFRR */

    /* call is to revert to glyph indices?  Done! */
    if ((platform == 0xFFFF) && (encoding == 0xFFFF))
    {
        STATE.error = SUCCESS;
        return 0;
    }
    /* locate the FULL cmap table in lfnt -- remember it's in big-endian byte order */

    if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
    {
        return 0;
    }
    ttf = (FS_VOID *)(lfnt->fnt);

    if (!ttf->memptr || (FS_VOID *)(ttf->decomp))
    {
        cmap = ttf->cmap;
    }
    else
    {
        cmap = ttf->cmap;
    }
    /* locate the proper sub-table */

    n_tabs = SWAPW(cmap->number);
    tabs = cmap->tables;
    offset = 0;
    for (i = 0; i < n_tabs; i++)
    {
        if ((platform == SWAPW(tabs[i].platform)) &&
            (encoding == SWAPW(tabs[i].encoding)))
        {
            /* byte offset to subtable */
            offset = SWAPL(tabs[i].offset);
            break;
        }
    }
    if (offset == 0)
    {
        STATE.error = ERR_CMAP_UNSUPPORTED;
        return 0;
    }
    else
    {
        STATE.error = SUCCESS;
        return offset;
    }
}

static FS_USHORT *
offset_to_cmap(_DS_ LFNT *lfnt, FS_ULONG offset)
{
    TTF *ttf;
    FS_BYTE *cmap;

    if (offset == 0)
    {
        return NULL;
    }
    if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
    {
        return NULL;
    }
    ttf = (FS_VOID *)(lfnt->fnt);

    cmap = (FS_BYTE *)ttf->cmap;

    return (FS_USHORT *)(offset + cmap);
}

/****************************************************************/
/*
 *    Platform / encoding substition rules go here.
 *    If the asked for cmap isn't available for this component then
 *    fall through this list looking for one that we do have.
 *    Full repetoire cmaps first then BMP only cmaps.
*/
#define NUM_SUBS 7

static FS_CONST FS_SHORT Substitutions[NUM_SUBS][2] = { {3, 10},
                                                        {0,  6},
                                                        {0,  4},
                                                        {3,  1},
                                                        {0,  3},
                                                        {0,  1},
                                                        {0,  0}  };

static 
FS_ULONG GetEncodingSubstitution(_DS_ LFNT* lfnt, FS_USHORT platform, FS_USHORT encoding)
{
    FS_ULONG offset=0;
    FS_USHORT currentSub = 0;

    /* we only substitute if the platform,encoding is in the substitution list */
    while (currentSub < NUM_SUBS)
    {
        if (platform == Substitutions[currentSub][0])
            if (encoding == Substitutions[currentSub][1])
                break; /* found it */
        currentSub++;
    }

    if (currentSub >= NUM_SUBS)  /* platform,encoding not in the list */
        return 0;

    /* search for a substitute Unicode cmap */
    currentSub = 0;
    while (currentSub < NUM_SUBS)
    {
        offset = LFNT_get_cmap_offset(_PS_ lfnt, Substitutions[currentSub][0], Substitutions[currentSub][1]);

        if (STATE.error == SUCCESS)
        {
            return offset;
        }
        currentSub++;
    }

    return offset;
}

/****************************************************************/
/* set apparatus to support specified character mapping */
FS_LONG map_font(_DS_ TFNT *tfnt, FS_USHORT platform, FS_USHORT encoding)
{
    LFNT *lfnt;
    STATE.error = SUCCESS;
    
    if (STATE.cur_typeset.fntset == 0)
    {
        return STATE.error = ERR_NO_CURRENT_FNTSET;
    }

    STATE.platform = 0xFFFF; 
    STATE.encoding = 0xFFFF;

    lfnt = (LFNT *)(tfnt->cfnt->lfnt);
    
    /* Try to get the requested cmap. */
    tfnt->cmap_offset = LFNT_get_cmap_offset(_PS_ lfnt, platform, encoding);

    if (STATE.error != SUCCESS)
    {
        if (platform != STATE.varPlatform && encoding != STATE.varEncoding) /* never substitute variant cmap */
        {
            /* try for a substitute cmap. */
            tfnt->cmap_offset = GetEncodingSubstitution(_PS_ lfnt, platform, encoding);

            /* If failed, return the 402 error. */
            if (STATE.error != SUCCESS)
            {
                return STATE.error;
            }
        }
        else
        {
            if (STATE.cur_typeset.capacity > 1)
                STATE.error = SUCCESS; /* not an error for linked fonts */
        }
    }

    STATE.platform = platform;
    STATE.encoding = encoding;

    return STATE.error;
}

static FS_BOOLEAN
CFNT_charInRange(UniRange upc, FS_ULONG id)
{
    FS_ULONG range;

    /* always return 1 if ranges are unknown */
    if ((upc.range1 == 0xFFFFFFFF) && (upc.range2 == 0xFFFFFFFF))
        return 1;

    /* check if Unicode id might be in this component font */

    range = id >> 10; /* range value */

    if (range < 32)
        return (upc.range1 & (1L << range)) ? 1 : 0;

    else if (range < 64)
        return (upc.range2 & (1L << (range - 32))) ? 1 : 0;

    else
        return 1;
}

/****************************************************************/
/* using CMAP translate glyph_id => glyph_index */
FS_USHORT map_char(_DS_ FS_ULONG id, FS_ULONG varSelector)
{
    TFNT *tfnt;
    FS_USHORT component;

    STATE.error = SUCCESS;

    /* must have a current fntset */
    if (STATE.cur_typeset.fntset == 0)
    {
        STATE.error = ERR_NO_CURRENT_FNTSET;
        return 0;
    }

    for (component = 0, tfnt = STATE.cur_typeset.tfntarray;
         component < STATE.cur_typeset.fntset->num_fonts;
         component++, tfnt++)
    {
        CFNT *cfnt;
        LFNT *lfnt;
        FS_USHORT result;

        cfnt = tfnt->cfnt;
        lfnt = (LFNT *)(cfnt->lfnt);
        if (!lfnt) return 0;

        if (!tfnt->cmap_offset)
        {
            map_font(_PS_ tfnt, STATE.platform, STATE.encoding);
            if(STATE.error)
                return 0;
        }

#ifdef FS_PFRR
        if (lfnt->fnt_type == PFR_TYPE)
        {
            STATE.cur_font = component;
            STATE.cur_lfnt = lfnt;
            STATE.cur_sfnt = tfnt->sfnt; /* no need to increment ref_count */

            STATE.error = SUCCESS;
            return (FS_USHORT)id;
        }
#endif /* FS_PFRR */

        /* temporarily don't use cmap ? or no cmap? */
        if ((STATE.flags & FLAGS_CMAP_OFF) ||
            ((STATE.platform == 0xFFFF) && (STATE.encoding == 0xFFFF)))
        {
            TTF *ttf;
            FS_LONG index = id - cfnt->start_index;
            if (index < 0)
            {
                STATE.error = ERR_BAD_GLYF_INDEX;
                return 0;
            }
            if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
            {
                return 0;
            }
            ttf = (FS_VOID *)(lfnt->fnt);

            if (index < ttf->maxp->numGlyphs)
            {
                STATE.cur_font = component;
                STATE.cur_lfnt = lfnt;
                STATE.cur_sfnt = tfnt->sfnt; /* no need to increment ref_count */

#ifdef FS_EDGE_RENDER
                STATE.adfGridFitType = tfnt->adfRenderAttrs.gridFitType;
#endif
                STATE.error = SUCCESS;
                return (FS_USHORT)index;
            }
#ifdef FS_ICONS
            else if (ttf->icon_offset)
            {
                if (index < (ttf->maxp->numGlyphs + (FS_LONG)ttf->num_icons))
                {
                    STATE.cur_font = component;
                    STATE.cur_lfnt = lfnt;
                    STATE.cur_sfnt = tfnt->sfnt; /* no need to increment ref_count */

#ifdef FS_EDGE_RENDER
                    STATE.adfGridFitType = tfnt->adfRenderAttrs.gridFitType;
#endif
                    STATE.error = SUCCESS;
                    return (FS_USHORT)index;
                }
            }
#endif /* FS_ICONS */
            if (component == (STATE.cur_typeset.fntset->num_fonts - 1))
            {
                STATE.error = ERR_BAD_GLYF_INDEX;
                return 0;
            }
            continue; /* move on to next cfnt */
        } /* temporarily don't use cmap ? or no cmap? */

        /* optimizations for standard encodings     */
        /* cmap caching and Unicode range checking  */
        if ((STATE.platform == 3) &&
            ((STATE.encoding == 1) || (STATE.encoding == 10)))
        {
            FS_BOOLEAN included;

#ifdef FS_CACHE_CMAP
            /* check if Unicode is in cmap cache for this font */
            if ((lfnt->cmap_cache != 0) && (id < 0x10000))
            {
                FS_USHORT entry;

                entry = CMAP_CACHE_SLOT(lfnt, id);

                if (lfnt->cmap_cache[entry].unicode == (FS_USHORT)id)
                {
                    if (lfnt->cmap_cache[entry].index == 0) /* entry cached as missing character */
                    {
                        continue; /* keep looking in other cfnts */
                    }
                    else
                    {
                        STATE.cur_font = component;
                        STATE.cur_lfnt = lfnt;
                        STATE.cur_sfnt = tfnt->sfnt; /* no need to increment ref_count */

#ifdef FS_EDGE_RENDER
                        STATE.adfGridFitType = tfnt->adfRenderAttrs.gridFitType;
#endif
                        STATE.error = SUCCESS;
                        return (lfnt->cmap_cache[entry].index); /* cache hit */
                    }
                }
            }
#endif /* FS_CACHE_CMAP */

            switch (id & 0xF0000) /* plane */
            {
            case 0x00000:
                included = CFNT_charInRange(cfnt->bmp, id & 0xFFFF);
                break;
            case 0x10000:
                included = CFNT_charInRange(cfnt->smp, id & 0xFFFF);
                break;
            case 0x20000:
                included = CFNT_charInRange(cfnt->sip, id & 0xFFFF);
                break;
            case 0xE0000:
                included = CFNT_charInRange(cfnt->ssp, id & 0xFFFF);
                break;
            default:
                included = 1;
                break;
            }
            if (!included)
            {
#ifdef FS_ICONS
                TTF *ttf;
                if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
                {
                    return 0;
                }
                ttf = (FS_VOID *)(lfnt->fnt);

                if (ttf && ttf->num_icons &&
                    (id >= ttf->icon_first) && (id <= ttf->icon_last) )
                {
                    result = map_char_icon(_PS_ ttf, id);
                    /* cache result and return, even if zero */
#ifdef FS_CACHE_CMAP
                    if ((lfnt->cmap_cache != 0) && (id < 0x10000))
                    {
                        FS_USHORT entry;

                        entry = CMAP_CACHE_SLOT(lfnt, id);

                        lfnt->cmap_cache[entry].unicode = (FS_USHORT)id;
                        lfnt->cmap_cache[entry].index = result; /* cache result */
                    }
#endif /* FS_CACHE_CMAP */
                    return result;
                }
                else
#endif /* FS_ICONS */
                    continue; /* move on to next cfnt */
            }
        }

        result = LFNT_map_char(_PS_ lfnt, tfnt->cmap_offset, id, varSelector);

#ifdef FS_ICONS
        if (!result)
        {
            TTF *ttf;
            ttf = (FS_VOID *)(lfnt->fnt);
            if (ttf && ttf->num_icons &&
                (id >= ttf->icon_first) && (id <= ttf->icon_last) ) /* try icon */
            {
                result = map_char_icon(_PS_ ttf, id);
            }
        }
#endif /* FS_ICONS */

#ifdef FS_CACHE_CMAP /* cache this result, even if 0 */
        if ((STATE.platform == 3) &&
            ((STATE.encoding == 1) || (STATE.encoding == 10)))
        {
            if ((lfnt->cmap_cache != 0) && (id < 0x10000))
            {
                FS_USHORT entry;

                entry = CMAP_CACHE_SLOT(lfnt, id);

                lfnt->cmap_cache[entry].unicode = (FS_USHORT)id;
                lfnt->cmap_cache[entry].index = result; /* cache result */
            }
        }
#endif /* FS_CACHE_CMAP */

        if (result)
        {
            STATE.cur_font = component;
            STATE.cur_lfnt = lfnt;
            STATE.cur_sfnt = tfnt->sfnt; /* no need to increment ref_count */

#ifdef FS_EDGE_RENDER
            STATE.adfGridFitType = tfnt->adfRenderAttrs.gridFitType;
#endif
            STATE.error = SUCCESS;
            return result;
        }
    } /* for each typeset element */

    /* id not found, return missing character index */
    /* from first element of current typeset        */
    STATE.cur_font = 0;
    STATE.cur_lfnt = (LFNT *)(STATE.cur_typeset.tfntarray->cfnt->lfnt);
    STATE.cur_sfnt = STATE.cur_typeset.tfntarray->sfnt; /* no need to increment ref_count */

#ifdef FS_EDGE_RENDER
    STATE.adfGridFitType = STATE.cur_typeset.tfntarray->adfRenderAttrs.gridFitType;
#endif
    return 0;
}

/* perform cmap search for specific LFNT */
FS_USHORT LFNT_map_char(_DS_ LFNT *lfnt, FS_ULONG cmap_offset, FS_ULONG id, FS_ULONG varSelector)
{
    FS_USHORT format, *cmap;
    FS_USHORT result;

    if (cmap_offset == 0)
    {
        cmap_offset = LFNT_get_cmap_offset(_PS_ lfnt, STATE.platform, STATE.encoding);
    }

    cmap = offset_to_cmap(_PS_ lfnt, cmap_offset);
    if (!cmap)
    {
        return 0;
    }
    format = SWAPW(cmap[0]);

    /* ? 16 bit charcode formats with too large an id */
    if ((id > 0xFFFFU) && (format < 8))
    {
        STATE.error = ERR_BAD_GLYF_INDEX;
        return 0;
    }
    /* call map_char_x function based on format */
    switch (format)
    {
    case  4:
        result = map_char_4(cmap, id);
        break;
    case 12:
        result = map_char_12(cmap, id);
        break;
    case  0:
        result = map_char_0(cmap, id);
        break;
    case  2:
        result = map_char_2(cmap, id);
        break;
    case  6:
        result = map_char_6(cmap, id);
        break;
    case  8:
        result = map_char_8(cmap, id);
        break;
    case  5:
        result = map_char_5(cmap, id);
        break;
    case  14:
        result = map_char_14(cmap, id, varSelector);
        break;
    default:
        result = 0;
        break;
    }
    return result;
}

/****************************************************************/
/* table of bytes indexed by a byte... nothing to swap */
static FS_USHORT map_char_0(FS_USHORT *cmap, FS_ULONG id)
{
    FS_BYTE *bytes = (FS_BYTE *)cmap;

    return (FS_USHORT)(id > 255) ? 0 : (FS_USHORT)bytes[6 + id];
}

/****************************************************************/
#define CMAP_FIRST  0
#define CMAP_COUNT  1
#define CMAP_DELTA  2
#define CMAP_OFFSET 3

static FS_USHORT map_char_2(FS_USHORT *cmap, FS_ULONG id)
{
    FS_USHORT *subHeaderKeys, *subHeaders, *subHeader;
    FS_USHORT first, count, offset;
    FS_SHORT delta;
    FS_LONG hi, lo, index;

    subHeaderKeys = cmap + 3;
    subHeaders = subHeaderKeys + 256;

    /* get the proper subHeader */
    lo = id & 0xFF;
    hi = 0xFF & (id >> 8);
    subHeader = subHeaders + hi;

    /* extract goodies */
    first = SWAPW(subHeader[CMAP_FIRST]);
    count = SWAPW(subHeader[CMAP_COUNT]);
    delta = (FS_SHORT) SWAPW(subHeader[CMAP_DELTA]);
    offset = SWAPW(subHeader[CMAP_OFFSET]);
    offset = offset >> 1;  /* byte offset ==> FS_SHORT offset */

    index = lo - first;
    if (index < 0 || index >= count)
        return 0;    /* not in this subHeader */

    index = SWAPW(subHeader[CMAP_OFFSET + offset + index]);
    if (index != 0)
        index += delta;
    return (FS_USHORT) index;
}

#undef CMAP_FIRST
#undef CMAP_COUNT
#undef CMAP_DELTA
#undef CMAP_OFFSET

/****************************************************************/
#define CMAP_COUNT 3
#define CMAP_SEL   5
#define CMAP_ENDS  7

static FS_USHORT map_char_4(FS_USHORT *cmap, FS_ULONG id)
{
    FS_USHORT index, count, range, sel, *m = cmap;
    FS_SHORT delta;
    FS_LONG offset;
    FS_USHORT *starts, *ends;

    /* number of segments */
    count = SWAPW(m[CMAP_COUNT]) >> 1;

    /* entrySelector ...  floor(log2(count)) */
    sel = SWAPW(m[CMAP_SEL]);

    /* convert to index */
    if (sel > 0)
        sel = 2 << (sel - 1);
    if (sel == count) /* can happen sometimes */
        sel >>= 1;

    /* point to endCount[] and startCount[] */
    ends = m + CMAP_ENDS;
    starts = ends + count + 1;

    /* in a new block to help the compiler generate better code */
    {
        register FS_USHORT t, *p = ends;
        register unsigned int idx = 0;

        t = GET_xWORD(p + sel);
        if (t < id) idx = count - sel - 1;
        sel >>= 1;

        switch (sel)
        {
        case 16384: t = GET_xWORD(p+idx+16383); if (t < id) idx += 16384;/* FALLTHROUGH */
        case  8192: t = GET_xWORD(p+idx+ 8191); if (t < id) idx +=  8192;/* FALLTHROUGH */
        case  4096: t = GET_xWORD(p+idx+ 4095); if (t < id) idx +=  4096;/* FALLTHROUGH */
        case  2048: t = GET_xWORD(p+idx+ 2047); if (t < id) idx +=  2048;/* FALLTHROUGH */
        case  1024: t = GET_xWORD(p+idx+ 1023); if (t < id) idx +=  1024;/* FALLTHROUGH */
        case   512: t = GET_xWORD(p+idx+  511); if (t < id) idx +=   512;/* FALLTHROUGH */
        case   256: t = GET_xWORD(p+idx+  255); if (t < id) idx +=   256;/* FALLTHROUGH */
        case   128: t = GET_xWORD(p+idx+  127); if (t < id) idx +=   128;/* FALLTHROUGH */
        case    64: t = GET_xWORD(p+idx+   63); if (t < id) idx +=    64;/* FALLTHROUGH */
        case    32: t = GET_xWORD(p+idx+   31); if (t < id) idx +=    32;/* FALLTHROUGH */
        case    16: t = GET_xWORD(p+idx+   15); if (t < id) idx +=    16;/* FALLTHROUGH */
        case     8: t = GET_xWORD(p+idx+    7); if (t < id) idx +=     8;/* FALLTHROUGH */
        case     4: t = GET_xWORD(p+idx+    3); if (t < id) idx +=     4;/* FALLTHROUGH */
        case     2: t = GET_xWORD(p+idx+    1); if (t < id) idx +=     2;
            break;
        default:
            break;
        }
        index = (FS_USHORT)idx;
    }

    /* now a small linear search to find first segment where id <= ends[index] */
    while (id > SWAPW(ends[index]))
        index++;

    /* now do the mapping for the index-th segment */
    offset = id - SWAPW(starts[index]);
    if (offset < 0)                    /* id is un-mapped */
        index = 0;
    else
    {
        m = starts + index + count;    /* m = & idDelta[index] */
        delta = SWAPW(*m);
        m += count;                    /* m = & idRangeOffset[index] */

        range = SWAPW(*m) >> 1;
        if (range == 0)
            index = (FS_USHORT)id + delta;
        else
        {
            m += range + offset;
            index = SWAPW(*m);
            if (index != 0)
                index += delta;
        }
    }
    return index;
}
/* Monotype Imaging proprietary version of format 4 with better compression */
/* Depends on CMAP_COUNT, CMAP_SEL, CMAP_ENDS (same as format 4)            */
static FS_USHORT map_char_5(FS_USHORT *cmap, FS_ULONG id)
{
    FS_USHORT index, count, range, sel, *m = cmap;
    FS_SHORT delta;
    FS_LONG offset;
    FS_USHORT *ends;
    FS_CHAR *starts;
    FS_USHORT st;

    /* number of segments */
    count = SWAPW(m[CMAP_COUNT]) >> 1;

    /* entrySelector ...  floor(log2(count)) */
    sel = SWAPW(m[CMAP_SEL]);

    /* convert to index */
    if (sel > 0)
        sel = 2 << (sel - 1);
    if (sel == count) /* can happen sometimes */
        sel >>= 1;

    /* point to endCount[] and startCount[] */
    ends = m + CMAP_ENDS;
    starts = (FS_CHAR *)(ends + count + 1);

    /* in a new block to help the compiler generate better code */
    {
        register FS_USHORT t, *p = ends;
        register unsigned int idx = 0;

        t = GET_xWORD(p + sel);
        if (t < id) idx = count - sel - 1;
        sel >>= 1;

        switch (sel)
        {
        case 16384: t = GET_xWORD(p+idx+16383); if (t < id) idx += 16384;/* FALLTHROUGH */
        case  8192: t = GET_xWORD(p+idx+ 8191); if (t < id) idx +=  8192;/* FALLTHROUGH */
        case  4096: t = GET_xWORD(p+idx+ 4095); if (t < id) idx +=  4096;/* FALLTHROUGH */
        case  2048: t = GET_xWORD(p+idx+ 2047); if (t < id) idx +=  2048;/* FALLTHROUGH */
        case  1024: t = GET_xWORD(p+idx+ 1023); if (t < id) idx +=  1024;/* FALLTHROUGH */
        case   512: t = GET_xWORD(p+idx+  511); if (t < id) idx +=   512;/* FALLTHROUGH */
        case   256: t = GET_xWORD(p+idx+  255); if (t < id) idx +=   256;/* FALLTHROUGH */
        case   128: t = GET_xWORD(p+idx+  127); if (t < id) idx +=   128;/* FALLTHROUGH */
        case    64: t = GET_xWORD(p+idx+   63); if (t < id) idx +=    64;/* FALLTHROUGH */
        case    32: t = GET_xWORD(p+idx+   31); if (t < id) idx +=    32;/* FALLTHROUGH */
        case    16: t = GET_xWORD(p+idx+   15); if (t < id) idx +=    16;/* FALLTHROUGH */
        case     8: t = GET_xWORD(p+idx+    7); if (t < id) idx +=     8;/* FALLTHROUGH */
        case     4: t = GET_xWORD(p+idx+    3); if (t < id) idx +=     4;/* FALLTHROUGH */
        case     2: t = GET_xWORD(p+idx+    1); if (t < id) idx +=     2;
            break;
        default:
            break;
        }
        index = (FS_USHORT)idx;
    }

    /* now a small linear search to find first segment where id <= ends[index] */
    while (id > SWAPW(ends[index]))
        index++;

    /* now do the mapping for the index-th segment */
    st = (SWAPW(ends[index])) - ABS(starts[index]) + 1;
    offset = id - st;
    if (offset < 0)                    /* id is un-mapped */
        index = 0;
    else
    {
        m = (FS_USHORT *)(starts + index * 2 + count);    /* m = & idDelta[index] */
        delta = SWAPW(*m);

        range = delta;
        if (starts[index] > 0)
            index = (FS_USHORT)id + delta;
        else
        {
            m += (range >> 1) + offset;
            index = SWAPW(*m);
        }
    }
    return index;
}

#undef CMAP_COUNT
#undef CMAP_SEL
#undef CMAP_ENDS

/****************************************************************/
/* thanks to MRR for algorithm correction */
#define CMAP_FIRST 3
#define CMAP_COUNT 4
#define CMAP_TABLE 5

static FS_USHORT map_char_6(FS_USHORT *cmap, FS_ULONG id)
{
    FS_LONG index;

    index = id - (FS_LONG) SWAPW(cmap[CMAP_FIRST]);
    if (index < 0 || index >= (FS_LONG)SWAPW(cmap[CMAP_COUNT]))
        return 0;
    else
        return SWAPW(cmap[CMAP_TABLE + index]);
}
#undef CMAP_FIRST
#undef CMAP_COUNT
#undef CMAP_TABLE

/****************************************************************/
/* the following table is floor(log2(n)) for n = 0 to 255   */
/* note that floor(log2(0)) is undefined but set to 0 below */
static FS_CONST FS_BYTE
floorLog2_256[] =
{
    0, 0, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7
};

/****************************************************************/
/* find the value of FLOOR(log2(count))                         */
/* this value is used as an entry selector in binary search     */
/* in format 8 and 12                                           */
static FS_USHORT
entrySelector(FS_ULONG n)
{
    FS_USHORT floorLog2; /* floor(log2(n)) */

    if      (n & 0xFFFF0000)    if      (n & 0xFF000000)    floorLog2 = 24 + floorLog2_256[n >> 24];
                                else /* (n & 0x00FF0000) */ floorLog2 = 16 + floorLog2_256[n >> 16];
    else /* (n & 0x0000FFFF) */ if      (n & 0x0000FF00)    floorLog2 =  8 + floorLog2_256[n >>  8];
                                else /* (n & 0x000000FF) */ floorLog2 =      floorLog2_256[n      ];

    return floorLog2;
}

#undef  GROUPSIZE
#define GROUPSIZE 3  /* group is 3 longs */

/****************************************************************/
static FS_USHORT map_char_8(FS_USHORT *cmap, FS_ULONG id)
{
    FS_ULONG    *m = (FS_ULONG*)cmap, *n;
    FS_ULONG    *m_last;
    FS_ULONG    numGroups, startCharCode, endCharCode, startGlyphID;
    FS_USHORT   sel, index;


    /* move to is32[] */
    m += 3;

    /* skip is32[] */
    m += ( 8192 / sizeof(FS_ULONG) );

    numGroups = (FS_ULONG)GET_xLONG(m);
    sel = entrySelector(numGroups);

    /* convert to index */
    if (sel > 0)
        sel = 2 << (sel - 1);
    if (sel == numGroups) /* can happen sometimes */
        sel >>= 1;

    m += 1;  /* point to first group */

    /* in a new block to help the compiler generate better code */
    {
        register FS_ULONG t, *q, *p = m + 1; /* point to endCharCode in first group */
        register unsigned int idx = 0;

        q = p + sel * GROUPSIZE;
        t = GET_xLONG(q);
        if (t < id) idx = (numGroups - sel - 1) * GROUPSIZE;
        sel >>= 1;

        switch (sel)
        {
        case 32768: q = p+idx+98301/*32767*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 98304/*32768*GROUPSIZE*/;/* FALLTHROUGH */
        case 16384: q = p+idx+49149/*16383*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 49152/*16384*GROUPSIZE*/;/* FALLTHROUGH */
        case  8192: q = p+idx+24573/* 8191*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 24576/* 8192*GROUPSIZE*/;/* FALLTHROUGH */
        case  4096: q = p+idx+12285/* 4095*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 12288/* 4096*GROUPSIZE*/;/* FALLTHROUGH */
        case  2048: q = p+idx+ 6141/* 2047*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=  6144/* 2048*GROUPSIZE*/;/* FALLTHROUGH */
        case  1024: q = p+idx+ 3069/* 1023*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=  3072/* 1024*GROUPSIZE*/;/* FALLTHROUGH */
        case   512: q = p+idx+ 1533/*  511*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=  1536/*  512*GROUPSIZE*/;/* FALLTHROUGH */
        case   256: q = p+idx+  765/*  255*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=   768/*  256*GROUPSIZE*/;/* FALLTHROUGH */
        case   128: q = p+idx+  381/*  127*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=   384/*  128*GROUPSIZE*/;/* FALLTHROUGH */
        case    64: q = p+idx+  189/*   63*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=   192/*   64*GROUPSIZE*/;/* FALLTHROUGH */
        case    32: q = p+idx+   93/*   31*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    96/*   32*GROUPSIZE*/;/* FALLTHROUGH */
        case    16: q = p+idx+   45/*   15*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    48/*   16*GROUPSIZE*/;/* FALLTHROUGH */
        case     8: q = p+idx+   21/*    7*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    24/*    8*GROUPSIZE*/;/* FALLTHROUGH */
        case     4: q = p+idx+    9/*    3*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    12/*    4*GROUPSIZE*/;/* FALLTHROUGH */
        case     2: q = p+idx+    3/*    1*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=     6/*    2*GROUPSIZE*/;/* FALLTHROUGH */
            break;
        default:
            break;
        }
        m_last = m + (numGroups - 1) * GROUPSIZE + 1; /* point to endCharCode in last possible group */
        m += idx + 1;      /* point to endCharCode in lowest group */
    }

    /* now a small linear search to find first segment where id <= endCharCode */
    for (endCharCode = GET_xLONG(m); ((id > endCharCode) && (m < m_last)); endCharCode = GET_xLONG(m))
        m += GROUPSIZE;

    /* id must be in this group or it's not found */
    n = m - 1;
    startCharCode = (FS_ULONG)GET_xLONG(n);
    if ((id < startCharCode) || (id > endCharCode))
        return 0;

    /* id must be in this group */
    n = m + 1;
    startGlyphID  = (FS_ULONG)GET_xLONG(n);

    index = (FS_USHORT)(id - startCharCode + startGlyphID);

    return index;
}

#undef  GROUPSIZE
#define GROUPSIZE 3  /* group is 3 longs */

/****************************************************************/
static FS_USHORT map_char_12(FS_USHORT *cmap, FS_ULONG id)
{
    FS_ULONG    *m = (FS_ULONG*)cmap, *n;
    FS_ULONG    *m_last;
    FS_ULONG    numGroups, startCharCode, endCharCode, startGlyphID;
    FS_USHORT   sel, index;

    /* move to number of groupings */
    m += 3;

    numGroups = (FS_ULONG)GET_xLONG(m);
    sel = entrySelector(numGroups);

    /* convert to index */
    if (sel > 0)
        sel = 2 << (sel - 1);
    if (sel == numGroups) /* can happen sometimes */
        sel >>= 1;

    m += 1;  /* point to first group */

    /* in a new block to help the compiler generate better code */
    {
        register FS_ULONG t, *q, *p = m + 1; /* point to endCharCode in first group */
        register unsigned int idx = 0;

        q = p + sel * GROUPSIZE;
        t = GET_xLONG(q);
        if (t < id) idx = (numGroups - sel - 1) * GROUPSIZE;
        sel >>= 1;

        switch (sel)
        {
        case 32768: q = p+idx+98301/*32767*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 98304/*32768*GROUPSIZE*/;/* FALLTHROUGH */
        case 16384: q = p+idx+49149/*16383*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 49152/*16384*GROUPSIZE*/;/* FALLTHROUGH */
        case  8192: q = p+idx+24573/* 8191*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 24576/* 8192*GROUPSIZE*/;/* FALLTHROUGH */
        case  4096: q = p+idx+12285/* 4095*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 12288/* 4096*GROUPSIZE*/;/* FALLTHROUGH */
        case  2048: q = p+idx+ 6141/* 2047*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=  6144/* 2048*GROUPSIZE*/;/* FALLTHROUGH */
        case  1024: q = p+idx+ 3069/* 1023*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=  3072/* 1024*GROUPSIZE*/;/* FALLTHROUGH */
        case   512: q = p+idx+ 1533/*  511*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=  1536/*  512*GROUPSIZE*/;/* FALLTHROUGH */
        case   256: q = p+idx+  765/*  255*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=   768/*  256*GROUPSIZE*/;/* FALLTHROUGH */
        case   128: q = p+idx+  381/*  127*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=   384/*  128*GROUPSIZE*/;/* FALLTHROUGH */
        case    64: q = p+idx+  189/*   63*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=   192/*   64*GROUPSIZE*/;/* FALLTHROUGH */
        case    32: q = p+idx+   93/*   31*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    96/*   32*GROUPSIZE*/;/* FALLTHROUGH */
        case    16: q = p+idx+   45/*   15*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    48/*   16*GROUPSIZE*/;/* FALLTHROUGH */
        case     8: q = p+idx+   21/*    7*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    24/*    8*GROUPSIZE*/;/* FALLTHROUGH */
        case     4: q = p+idx+    9/*    3*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    12/*    4*GROUPSIZE*/;/* FALLTHROUGH */
        case     2: q = p+idx+    3/*    1*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=     6/*    2*GROUPSIZE*/;/* FALLTHROUGH */
            break;
        default:
            break;
        }
        m_last = m + (numGroups - 1) * GROUPSIZE + 1; /* point to endCharCode in last possible group */
        m += idx + 1;      /* point to endCharCode in lowest group */
    }

    /* now a small linear search to find first segment where id <= endCharCode */
    for (endCharCode = GET_xLONG(m); ((id > endCharCode) && (m < m_last)); endCharCode = GET_xLONG(m))
        m += GROUPSIZE;

    /* id must be in this group or it's not found */
    n = m - 1;
    startCharCode = (FS_ULONG)GET_xLONG(n);
    if ((id < startCharCode) || (id > endCharCode))
        return 0;

    /* id must be in this group */
    n = m + 1;
    startGlyphID  = (FS_ULONG)GET_xLONG(n);

    index = (FS_USHORT)(id - startCharCode + startGlyphID);

    return index;
}

/****************************************************************/
static FS_USHORT map_char_14(FS_USHORT *cmap, FS_ULONG id, FS_ULONG varSelector)
{
    FS_BYTE   *p = (FS_BYTE *)cmap;
    FS_ULONG   numVarSelectorRecords;

    p += 6; /* skip past format and length */
    numVarSelectorRecords = (FS_ULONG)GET_xLONG((FS_ULONG *)p);
    if (numVarSelectorRecords == 0) return 0;

    p += 4; /* points to first variation selection record */

    /* find variation selector if present in table */
    while (numVarSelectorRecords--)
    {
        FS_ULONG selector = GET_xUINT24(p);
        if (selector == varSelector)
        {
            FS_ULONG nonDefaultUVSOffset;
            p += 7; /* skip past varSelector and defaultUVSOffset */
            nonDefaultUVSOffset = (FS_ULONG)GET_xLONG((FS_ULONG *)p);
            if (nonDefaultUVSOffset)
            {
                FS_ULONG numUVSMappings;
                p = (FS_BYTE *)cmap + nonDefaultUVSOffset;
                numUVSMappings = (FS_ULONG)GET_xLONG((FS_ULONG *)p);
                p += 4; /* points to first unicode variation selector record */
                while (numUVSMappings--)
                {
                    FS_ULONG unicode = GET_xUINT24(p);
                    if (unicode == id) /* found non-default variant */
                    {
                        FS_USHORT glyphindex;
                        p += 3;
                        glyphindex = (FS_USHORT)GET_xWORD(p);
                        return glyphindex;
                    }
                    if (unicode > id)
                        break;
                    p += 5; /* skip past UINT24 + USHORT */
                }
            }
            return 0;
        }
        else
            p += 11; /* skip past UINT24 + 2 x ULONG */
    }

    return 0;
}

/****************************************************************/
/* using CMAP translate glyph index to unicode */
static FS_ULONG inverse_map_char_tfnt(_DS_ FS_ULONG gi, TFNT *tfnt)
{
    FS_USHORT *cmap;
    CFNT *cfnt;
    LFNT *lfnt;

    cfnt = tfnt->cfnt;
    lfnt = (LFNT *)(cfnt->lfnt);
    if (lfnt == 0)
    {
        STATE.error = ERR_NO_CURRENT_LFNT;
        return 0xFFFF;
    }

    if (!tfnt->cmap_offset)
    {
        map_font(_PS_ tfnt, STATE.platform, STATE.encoding);
        if (STATE.error)
            return 0xFFFF;
    }

#ifdef FS_PFRR
    if (lfnt->fnt_type == PFR_TYPE)
    {
        STATE.error = SUCCESS;
        return (FS_USHORT)gi;
    }
#endif /* FS_PFRR */

    if (gi <= cfnt->start_index)
    {
        STATE.error = ERR_BAD_GLYF_INDEX;
        return 0xFFFF;
    }

    cmap = offset_to_cmap(_PS_ lfnt, tfnt->cmap_offset);
    if (!cmap)
    {
        STATE.error = ERR_INVALID_CMAP;
        return 0xFFFF;
    }

    if (SWAPW(cmap[0]) == 4)
        return inverse_map_char_4(cmap, (FS_USHORT)(gi - cfnt->start_index));
    else if (SWAPW(cmap[0]) == 12)
        return inverse_map_char_12(cmap, gi - cfnt->start_index);

    return 0xFFFF;
}

FS_ULONG inverse_map_char(_DS_ FS_ULONG gi, TFNT *current_tfnt)
{
    TFNT *tfnt;
    FS_USHORT component;
    FS_ULONG result = 0xFFFF;

    if (STATE.platform != 3 || (STATE.encoding != 1 && STATE.encoding != 10))
    {
        return 0xFFFF;
    }

    /* must have a current fntset */
    if (STATE.cur_typeset.fntset == 0)
    {
        STATE.error = ERR_NO_CURRENT_FNTSET;
        return 0;
    }

    /* if the user explicitly passed in a current TFNT - use it */
    if (current_tfnt != NULL)
    {
        result = inverse_map_char_tfnt(_PS_ gi, current_tfnt);
    } 
    else 
    {
        /* search each tfnt/cfnt */
        for (component = 0, tfnt = STATE.cur_typeset.tfntarray;
             component < STATE.cur_typeset.fntset->num_fonts;
             component++, tfnt++)
        {
            result = inverse_map_char_tfnt(_PS_ gi, tfnt);
            if (result != 0xFFFF || STATE.error)
                break; /* found Unicode or error */
        } /* for each tfnt/cfnt font */
    }

    if (STATE.error)
        return 0xFFFF;
    else
        return result;
}

/*lint -e578  Warning 578: Declaration of symbol '' hides symbol */

/****************************************************************/
/* reverse unicode lookup */
#define CMAP_COUNT 3
#define CMAP_ENDS  7

static FS_ULONG inverse_map_char_4(FS_USHORT *cmap, FS_USHORT gi)
{
    FS_USHORT count, *m = cmap;
    FS_USHORT i, j, k;
    FS_USHORT *startcode, *endcode, *delta, *range, *gidarray;
    FS_USHORT g, start, end;

    if (cmap == 0 || gi == 0)
        return 0;

    /* number of segments */
    count = SWAPW(m[CMAP_COUNT]) >> 1;

    /* point to endCount[] and startCount[] */
    endcode = m + CMAP_ENDS;
    startcode = endcode + count + 1;
    delta = startcode + count;
    range = delta + count;

    for ( i = 0; i < count; i++)
    {
        if (range[i] == 0)
        {
            /* find start and end glyph index of contiguous range */
            end = (SWAPW(endcode[i]) + SWAPW(delta[i])) % 65536;
            start = (SWAPW(startcode[i]) + SWAPW(delta[i])) % 65536;
            if (gi >= start && gi <= end)
            {
                g = (gi - SWAPW(delta[i])) % 65536;
                return g; /* return unicode value */
            }
        }
        else
        {
            /* linearly search gidarray */
            start = SWAPW(startcode[i]);
            end = SWAPW(endcode[i]);
            gidarray = ((FS_USHORT)(SWAPW(range[i])) >> 1) + &range[i];
            if (start == 0xFFFF) return start;
            for (k = 0, j = start; j <= end; j++, k++)
            {
                g = SWAPW(gidarray[k]);
                if (gi == g)
                    return j;
            }
        }
    }
    return 0xFFFF; /* not found */
}

static FS_ULONG inverse_map_char_12(FS_USHORT *cmap, FS_ULONG id)
{
    FS_ULONG    *m = (FS_ULONG*)cmap;
    FS_ULONG    *m_last;
    FS_ULONG    numGroups, startCharCode, endCharCode, startGlyphID;
    FS_USHORT   sel;

    if (cmap == 0 || id == 0)
        return 0;

    /* move to number of groupings */
    m += 3;

    numGroups = (FS_ULONG)GET_xLONG(m);
    sel = entrySelector(numGroups);

    /* convert to index */
    if (sel > 0)
        sel = 2 << (sel - 1);
    if (sel == numGroups) /* can happen sometimes */
        sel >>= 1;

    m += 1;  /* point to first group */

    /* in a new block to help the compiler generate better code */
    {
        register FS_ULONG t, *q, *p = m + 1; /* point to endCharCode in first group */
        register unsigned int idx = 0;

        q = p + sel * GROUPSIZE;
        t = GET_xLONG(q);
        if (t < id) idx = (numGroups - sel - 1) * GROUPSIZE;
        sel >>= 1;

        switch (sel)
        {
        case 32768: q = p+idx+98301/*32767*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 98304/*32768*GROUPSIZE*/;/* FALLTHROUGH */
        case 16384: q = p+idx+49149/*16383*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 49152/*16384*GROUPSIZE*/;/* FALLTHROUGH */
        case  8192: q = p+idx+24573/* 8191*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 24576/* 8192*GROUPSIZE*/;/* FALLTHROUGH */
        case  4096: q = p+idx+12285/* 4095*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx += 12288/* 4096*GROUPSIZE*/;/* FALLTHROUGH */
        case  2048: q = p+idx+ 6141/* 2047*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=  6144/* 2048*GROUPSIZE*/;/* FALLTHROUGH */
        case  1024: q = p+idx+ 3069/* 1023*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=  3072/* 1024*GROUPSIZE*/;/* FALLTHROUGH */
        case   512: q = p+idx+ 1533/*  511*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=  1536/*  512*GROUPSIZE*/;/* FALLTHROUGH */
        case   256: q = p+idx+  765/*  255*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=   768/*  256*GROUPSIZE*/;/* FALLTHROUGH */
        case   128: q = p+idx+  381/*  127*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=   384/*  128*GROUPSIZE*/;/* FALLTHROUGH */
        case    64: q = p+idx+  189/*   63*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=   192/*   64*GROUPSIZE*/;/* FALLTHROUGH */
        case    32: q = p+idx+   93/*   31*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    96/*   32*GROUPSIZE*/;/* FALLTHROUGH */
        case    16: q = p+idx+   45/*   15*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    48/*   16*GROUPSIZE*/;/* FALLTHROUGH */
        case     8: q = p+idx+   21/*    7*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    24/*    8*GROUPSIZE*/;/* FALLTHROUGH */
        case     4: q = p+idx+    9/*    3*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=    12/*    4*GROUPSIZE*/;/* FALLTHROUGH */
        case     2: q = p+idx+    3/*    1*GROUPSIZE*/; t = GET_xLONG(q); if (t < id) idx +=     6/*    2*GROUPSIZE*/;/* FALLTHROUGH */
            break;
        default:
            break;
        }
        m_last = m + (numGroups - 1) * GROUPSIZE + 1; /* point to endCharCode in last possible group */
        m += idx;      /* point to startCharCode in lowest group */
    }

    /* now a small linear search to find first segment where id <= endCharCode */
    while ( m < m_last )
    {
        startCharCode = GET_xLONG(m);
        endCharCode = GET_xLONG(m + 1);
        startGlyphID = GET_xLONG(m + 2);
        if (startGlyphID <= id && id <= (startGlyphID + endCharCode - startCharCode))
            return (startCharCode + id - startGlyphID);

        m += GROUPSIZE;
    }

    return 0xFFFF; /* not found */
}

/*lint +e578  Warning 578: Declaration of symbol '' hides symbol */

/**
* conduct a binary search for an icon key, ignoring size
*/
#ifdef FS_ICONS
static FS_ULONG icon_bin_search(FS_ULONG len, FS_ULONG key, FS_ULONG *vec)
{
    FS_LONG lo, mid, hi;
    FS_ULONG index;

    lo = 0;
    hi = len - 1;
    while (lo <= hi)
    {
        mid = (lo + hi) / 2;
        index = (SWAPL(vec[mid])) & 0xFFFF0000;
        if (index == key)
            return mid;
        if (index > key)
            hi = mid - 1;
        else
            lo = mid + 1;
    }
    return len;    /* failure */
}

/**
 * search icon table for id
 */
static FS_USHORT map_char_icon(_DS_ TTF *ttf, FS_ULONG id)
{
    FS_USHORT index = 0;

    if (ttf && ttf->icon_offset)
    {
        FS_ULONG offset, num, key, *indices;

        /* see how many icons there are */
        num = ttf->num_icons;
        offset = ttf->icon_offset + 12;

        /* read indices array */
#ifdef FS_MEM_DBG
        STATE.memdbgid = "icon(indices)";
#endif
        indices = (FS_ULONG *)ttf_read(_PS_ ttf, offset, 4 * num);

        /* search for any icon with id */
        key = id << 16;
        index = (FS_USHORT)icon_bin_search(num, key, indices);

        FSS_free(_PS_ indices);
        if (index == num)
            index = 0;
        else
            index += ttf->maxp->numGlyphs;
    }
    return index;
}
#endif /* FS_ICONS */

/****************************************************************/
